/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.server.coordinator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.druid.client.ImmutableDruidServer;
import org.apache.druid.java.util.emitter.EmittingLogger;
import org.apache.druid.server.coordination.ServerType;
import org.apache.druid.server.coordinator.SegmentCountsPerInterval;
import org.apache.druid.server.coordinator.loading.LoadQueuePeon;
import org.apache.druid.server.coordinator.loading.SegmentAction;
import org.apache.druid.server.coordinator.loading.SegmentHolder;
import org.apache.druid.timeline.DataSegment;
import org.apache.druid.timeline.SegmentId;

public class ServerHolder
implements Comparable<ServerHolder> {
    private static final Comparator<ServerHolder> MORE_AVAILABLE_SIZE_SERVER_FIRST = Comparator.comparing(ServerHolder::getAvailableSize).thenComparing(holder -> holder.getServer().getHost()).thenComparing(holder -> holder.getServer().getTier()).thenComparing(holder -> holder.getServer().getType()).reversed();
    private static final EmittingLogger log = new EmittingLogger(ServerHolder.class);
    private final ImmutableDruidServer server;
    private final LoadQueuePeon peon;
    private final boolean isDecommissioning;
    private final int maxAssignmentsInRun;
    private final int maxLifetimeInQueue;
    private final int movingSegmentCount;
    private final int loadingReplicaCount;
    private int totalAssignmentsInRun;
    private long sizeOfLoadingSegments;
    private long sizeOfDroppingSegments;
    private final Map<DataSegment, SegmentAction> queuedSegments = new HashMap<DataSegment, SegmentAction>();
    private final SegmentCountsPerInterval projectedSegments = new SegmentCountsPerInterval();

    public ServerHolder(ImmutableDruidServer server, LoadQueuePeon peon) {
        this(server, peon, false, 0, 1);
    }

    public ServerHolder(ImmutableDruidServer server, LoadQueuePeon peon, boolean isDecommissioning) {
        this(server, peon, isDecommissioning, 0, 1);
    }

    public ServerHolder(ImmutableDruidServer server, LoadQueuePeon peon, boolean isDecommissioning, int maxSegmentsInLoadQueue, int maxLifetimeInQueue) {
        this.server = server;
        this.peon = peon;
        this.isDecommissioning = isDecommissioning;
        this.maxAssignmentsInRun = maxSegmentsInLoadQueue == 0 ? Integer.MAX_VALUE : maxSegmentsInLoadQueue - peon.getSegmentsToLoad().size();
        this.maxLifetimeInQueue = maxLifetimeInQueue;
        AtomicInteger movingSegmentCount = new AtomicInteger();
        AtomicInteger loadingReplicaCount = new AtomicInteger();
        this.initializeQueuedSegments(movingSegmentCount, loadingReplicaCount);
        this.movingSegmentCount = movingSegmentCount.get();
        this.loadingReplicaCount = loadingReplicaCount.get();
    }

    private void initializeQueuedSegments(AtomicInteger movingSegmentCount, AtomicInteger loadingReplicaCount) {
        for (DataSegment dataSegment : this.server.iterateAllSegments()) {
            this.projectedSegments.addSegment(dataSegment);
        }
        ArrayList<SegmentHolder> expiredSegments = new ArrayList<SegmentHolder>();
        for (SegmentHolder holder : this.peon.getSegmentsInQueue()) {
            int runsInQueue = holder.incrementAndGetRunsInQueue();
            if (runsInQueue > this.maxLifetimeInQueue) {
                expiredSegments.add(holder);
            }
            SegmentAction action = holder.getAction();
            this.addToQueuedSegments(holder.getSegment(), this.simplify(action));
            if (action == SegmentAction.MOVE_TO) {
                movingSegmentCount.incrementAndGet();
            }
            if (action != SegmentAction.REPLICATE) continue;
            loadingReplicaCount.incrementAndGet();
        }
        for (DataSegment segment : this.peon.getSegmentsMarkedToDrop()) {
            this.addToQueuedSegments(segment, SegmentAction.MOVE_FROM);
        }
        if (!expiredSegments.isEmpty()) {
            ArrayList<SegmentHolder> arrayList = expiredSegments.size() > 10 ? expiredSegments.subList(0, 10) : expiredSegments;
            log.makeAlert("Load queue for server [%s], tier [%s] has [%d] segments stuck.", new Object[]{this.server.getName(), this.server.getTier(), expiredSegments.size()}).addData("segments", (Object)((Object)arrayList).toString()).addData("maxLifetime", (Object)this.maxLifetimeInQueue).emit();
        }
    }

    public ImmutableDruidServer getServer() {
        return this.server;
    }

    public LoadQueuePeon getPeon() {
        return this.peon;
    }

    public long getMaxSize() {
        return this.server.getMaxSize();
    }

    public long getSizeUsed() {
        return this.server.getCurrSize() + this.sizeOfLoadingSegments - this.sizeOfDroppingSegments;
    }

    public double getPercentUsed() {
        return 100.0 * (double)this.getSizeUsed() / (double)this.getMaxSize();
    }

    public boolean isDecommissioning() {
        return this.isDecommissioning;
    }

    public boolean isLoadQueueFull() {
        return this.totalAssignmentsInRun >= this.maxAssignmentsInRun;
    }

    public long getAvailableSize() {
        return this.getMaxSize() - this.getSizeUsed();
    }

    public boolean canLoadSegment(DataSegment segment) {
        return !this.isDecommissioning && !this.hasSegmentLoaded(segment.getId()) && this.getActionOnSegment(segment) == null && this.totalAssignmentsInRun < this.maxAssignmentsInRun && this.getAvailableSize() >= segment.getSize();
    }

    public SegmentAction getActionOnSegment(DataSegment segment) {
        return this.queuedSegments.get(segment);
    }

    public Map<DataSegment, SegmentAction> getQueuedSegments() {
        return new HashMap<DataSegment, SegmentAction>(this.queuedSegments);
    }

    public SegmentCountsPerInterval getProjectedSegments() {
        return this.projectedSegments;
    }

    public boolean isProjectedSegment(DataSegment segment) {
        SegmentAction action = this.getActionOnSegment(segment);
        if (action == null) {
            return this.hasSegmentLoaded(segment.getId());
        }
        return action.isLoad();
    }

    public List<DataSegment> getLoadingSegments() {
        ArrayList<DataSegment> loadingSegments = new ArrayList<DataSegment>();
        this.queuedSegments.forEach((segment, action) -> {
            if (action == SegmentAction.LOAD) {
                loadingSegments.add((DataSegment)segment);
            }
        });
        return loadingSegments;
    }

    public Collection<DataSegment> getServedSegments() {
        return this.server.iterateAllSegments();
    }

    public boolean isServingSegment(DataSegment segment) {
        return this.hasSegmentLoaded(segment.getId()) && this.getActionOnSegment(segment) == null;
    }

    public boolean isLoadingSegment(DataSegment segment) {
        return this.getActionOnSegment(segment) == SegmentAction.LOAD;
    }

    public boolean isDroppingSegment(DataSegment segment) {
        return this.getActionOnSegment(segment) == SegmentAction.DROP;
    }

    public int getNumMovingSegments() {
        return this.movingSegmentCount;
    }

    public int getNumLoadingReplicas() {
        return this.loadingReplicaCount;
    }

    public int getNumQueuedSegments() {
        return this.queuedSegments.size();
    }

    public boolean startOperation(SegmentAction action, DataSegment segment) {
        if (this.queuedSegments.containsKey(segment)) {
            return false;
        }
        if (action.isLoad()) {
            ++this.totalAssignmentsInRun;
        }
        this.addToQueuedSegments(segment, this.simplify(action));
        return true;
    }

    public boolean cancelOperation(SegmentAction action, DataSegment segment) {
        SegmentAction queuedAction = this.queuedSegments.get(segment);
        if (queuedAction != this.simplify(action)) {
            return false;
        }
        if (queuedAction == SegmentAction.MOVE_FROM || this.peon.cancelOperation(segment)) {
            this.removeFromQueuedSegments(segment, queuedAction);
            return true;
        }
        return false;
    }

    private boolean hasSegmentLoaded(SegmentId segmentId) {
        return this.server.getSegment(segmentId) != null;
    }

    public boolean isRealtimeServer() {
        return this.server.getType() == ServerType.REALTIME || this.server.getType() == ServerType.INDEXER_EXECUTOR;
    }

    private SegmentAction simplify(SegmentAction action) {
        return action == SegmentAction.REPLICATE ? SegmentAction.LOAD : action;
    }

    private void addToQueuedSegments(DataSegment segment, SegmentAction action) {
        this.queuedSegments.put(segment, action);
        if (action.isLoad()) {
            this.projectedSegments.addSegment(segment);
            this.sizeOfLoadingSegments += segment.getSize();
        } else {
            this.projectedSegments.removeSegment(segment);
            if (action == SegmentAction.DROP) {
                this.sizeOfDroppingSegments += segment.getSize();
            }
        }
    }

    private void removeFromQueuedSegments(DataSegment segment, SegmentAction action) {
        this.queuedSegments.remove(segment);
        if (action.isLoad()) {
            this.projectedSegments.removeSegment(segment);
            this.sizeOfLoadingSegments -= segment.getSize();
        } else {
            this.projectedSegments.addSegment(segment);
            if (action == SegmentAction.DROP) {
                this.sizeOfDroppingSegments -= segment.getSize();
            }
        }
    }

    @Override
    public int compareTo(ServerHolder serverHolder) {
        return MORE_AVAILABLE_SIZE_SERVER_FIRST.compare(this, serverHolder);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ServerHolder that = (ServerHolder)o;
        return Objects.equals(this.server.getHost(), that.server.getHost()) && Objects.equals(this.server.getTier(), that.server.getTier()) && Objects.equals((Object)this.server.getType(), (Object)that.server.getType());
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.server.getHost(), this.server.getTier(), this.server.getType()});
    }

    public String toString() {
        return "ServerHolder{" + this.server.getHost() + "}";
    }
}

