/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.faulttolerance.core.retry;

import io.smallrye.faulttolerance.core.Completer;
import io.smallrye.faulttolerance.core.FailureContext;
import io.smallrye.faulttolerance.core.FaultToleranceContext;
import io.smallrye.faulttolerance.core.FaultToleranceStrategy;
import io.smallrye.faulttolerance.core.Future;
import io.smallrye.faulttolerance.core.retry.AsyncDelay;
import io.smallrye.faulttolerance.core.retry.RetryEvents;
import io.smallrye.faulttolerance.core.retry.RetryLogger;
import io.smallrye.faulttolerance.core.retry.SyncDelay;
import io.smallrye.faulttolerance.core.retry.SyncDelayAsAsync;
import io.smallrye.faulttolerance.core.stopwatch.RunningStopwatch;
import io.smallrye.faulttolerance.core.stopwatch.Stopwatch;
import io.smallrye.faulttolerance.core.util.ExceptionDecision;
import io.smallrye.faulttolerance.core.util.Preconditions;
import io.smallrye.faulttolerance.core.util.ResultDecision;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.eclipse.microprofile.faulttolerance.exceptions.FaultToleranceException;

public class Retry<V>
implements FaultToleranceStrategy<V> {
    private final FaultToleranceStrategy<V> delegate;
    private final String description;
    private final ResultDecision resultDecision;
    private final ExceptionDecision exceptionDecision;
    private final long maxRetries;
    private final long maxTotalDurationInMillis;
    private final Supplier<SyncDelay> syncDelayBetweenRetries;
    private final Supplier<AsyncDelay> asyncDelayBetweenRetries;
    private final Stopwatch stopwatch;
    private final Consumer<FailureContext> beforeRetry;

    public Retry(FaultToleranceStrategy<V> delegate, String description, ResultDecision resultDecision, ExceptionDecision exceptionDecision, long maxRetries, long maxTotalDurationInMillis, Supplier<SyncDelay> syncDelayBetweenRetries, Supplier<AsyncDelay> asyncDelayBetweenRetries, Stopwatch stopwatch, Consumer<FailureContext> beforeRetry) {
        this.delegate = Preconditions.checkNotNull(delegate, "Retry delegate must be set");
        this.description = Preconditions.checkNotNull(description, "Retry description must be set");
        this.resultDecision = Preconditions.checkNotNull(resultDecision, "Result decision must be set");
        this.exceptionDecision = Preconditions.checkNotNull(exceptionDecision, "Exception decision must be set");
        this.maxRetries = maxRetries < 0L ? Long.MAX_VALUE : maxRetries;
        this.maxTotalDurationInMillis = maxTotalDurationInMillis <= 0L ? Long.MAX_VALUE : maxTotalDurationInMillis;
        this.syncDelayBetweenRetries = Preconditions.checkNotNull(syncDelayBetweenRetries, "Synchronous delay must be set");
        this.asyncDelayBetweenRetries = Preconditions.checkNotNull(asyncDelayBetweenRetries, "Asynchronous delay must be set");
        this.stopwatch = Preconditions.checkNotNull(stopwatch, "Stopwatch must be set");
        this.beforeRetry = beforeRetry;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Future<V> apply(FaultToleranceContext<V> ctx) {
        RetryLogger.LOG.trace("Retry started");
        try {
            AsyncDelay delay = ctx.isAsync() ? this.asyncDelayBetweenRetries.get() : new SyncDelayAsAsync(this.syncDelayBetweenRetries.get());
            RunningStopwatch runningStopwatch = this.stopwatch.start();
            Future<V> future = this.retryLoop(ctx, runningStopwatch, delay);
            return future;
        }
        finally {
            RetryLogger.LOG.trace("Retry finished");
        }
    }

    private Future<V> retryLoop(FaultToleranceContext<V> ctx, RunningStopwatch stopwatch, AsyncDelay delay) {
        Future<State> future = Future.loop(State.initial(), State::shouldContinue, state -> {
            if (state.attempt == 0) {
                return this.retryLoopIteration(ctx, stopwatch, (State<V>)state);
            }
            if ((long)state.attempt <= this.maxRetries) {
                if (stopwatch.elapsedTimeInMillis() >= this.maxTotalDurationInMillis) {
                    ctx.fireEvent(RetryEvents.Finished.MAX_DURATION_REACHED);
                    if (state.lastFailure != null) {
                        return Future.ofError(state.lastFailure);
                    }
                    return Future.ofError((Throwable)new FaultToleranceException(this.description + " reached max retry duration"));
                }
                RetryLogger.LOG.debugf("%s invocation failed, retrying (%d/%d)", this.description, state.attempt, this.maxRetries);
                ctx.fireEvent(RetryEvents.Retried.INSTANCE);
                Completer result = Completer.create();
                try {
                    delay.after(state.lastFailure, () -> this.retryLoopIteration(ctx, stopwatch, (State<V>)state).thenComplete(result), ctx.get(Executor.class));
                }
                catch (Exception e) {
                    if (ctx.isSync() && Thread.interrupted()) {
                        result.completeWithError(new InterruptedException());
                    }
                    result.completeWithError(e);
                }
                return result.future();
            }
            ctx.fireEvent(RetryEvents.Finished.MAX_RETRIES_REACHED);
            if (state.lastFailure != null) {
                return Future.ofError(state.lastFailure);
            }
            return Future.ofError((Throwable)new FaultToleranceException(this.description + " reached max retries"));
        });
        Completer completer = Completer.create();
        future.then((value, error) -> {
            if (error == null) {
                completer.complete(value.value);
            } else {
                completer.completeWithError((Throwable)error);
            }
        });
        return completer.future();
    }

    private Future<State<V>> retryLoopIteration(FaultToleranceContext<V> ctx, RunningStopwatch stopwatch, State<V> state) {
        if (stopwatch.elapsedTimeInMillis() >= this.maxTotalDurationInMillis) {
            ctx.fireEvent(RetryEvents.Finished.MAX_DURATION_REACHED);
            if (state.lastFailure != null) {
                return Future.ofError(state.lastFailure);
            }
            return Future.ofError((Throwable)new FaultToleranceException(this.description + " reached max retry duration"));
        }
        if (this.beforeRetry != null && state.attempt > 0) {
            try {
                this.beforeRetry.accept(new FailureContext(state.lastFailure, ctx));
            }
            catch (Exception e) {
                RetryLogger.LOG.warn("Before retry action has thrown an exception", e);
            }
        }
        Completer<State<V>> result = Completer.create();
        try {
            this.delegate.apply(ctx).then((value, error) -> {
                if (ctx.isSync()) {
                    if (error instanceof InterruptedException) {
                        ctx.fireEvent(RetryEvents.Finished.EXCEPTION_NOT_RETRYABLE);
                        result.completeWithError((Throwable)error);
                        return;
                    }
                    if (Thread.interrupted()) {
                        ctx.fireEvent(RetryEvents.Finished.EXCEPTION_NOT_RETRYABLE);
                        result.completeWithError(new InterruptedException());
                        return;
                    }
                }
                if (error == null) {
                    if (this.resultDecision.isConsideredExpected(value)) {
                        ctx.fireEvent(RetryEvents.Finished.VALUE_RETURNED);
                        result.complete(State.done(value));
                    } else {
                        result.complete(state.retry((Throwable)error));
                    }
                } else if (this.exceptionDecision.isConsideredExpected((Throwable)error)) {
                    ctx.fireEvent(RetryEvents.Finished.EXCEPTION_NOT_RETRYABLE);
                    result.completeWithError((Throwable)error);
                } else {
                    result.complete(state.retry((Throwable)error));
                }
            });
        }
        catch (Throwable e) {
            if (this.exceptionDecision.isConsideredExpected(e)) {
                ctx.fireEvent(RetryEvents.Finished.EXCEPTION_NOT_RETRYABLE);
                result.completeWithError(e);
            }
            result.complete(state.retry(e));
        }
        return result.future();
    }

    private static class State<V> {
        private final boolean shouldContinue;
        private final V value;
        private final int attempt;
        private final Throwable lastFailure;

        static <V> State<V> initial() {
            return new State<Object>(true, null, 0, null);
        }

        static <V> State<V> done(V value) {
            return new State<V>(false, value, -1, null);
        }

        private State(boolean shouldContinue, V value, int attempt, Throwable lastFailure) {
            this.shouldContinue = shouldContinue;
            this.value = value;
            this.attempt = attempt;
            this.lastFailure = lastFailure;
        }

        public boolean shouldContinue() {
            return this.shouldContinue;
        }

        public State<V> retry(Throwable failure) {
            return new State<Object>(true, null, this.attempt + 1, failure);
        }
    }
}

