/*
 * Decompiled with CFR 0.152.
 */
package org.apache.inlong.tubemq.server.common.zookeeper;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import org.apache.inlong.tubemq.server.common.utils.Bytes;
import org.apache.inlong.tubemq.server.common.zookeeper.RetryCounter;
import org.apache.inlong.tubemq.server.common.zookeeper.RetryCounterFactory;
import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RecoverableZooKeeper {
    private static final Logger logger = LoggerFactory.getLogger(RecoverableZooKeeper.class);
    private final RetryCounterFactory retryCounterFactory;
    private ZooKeeper zk;
    private Watcher watcher;
    private int sessionTimeout;
    private String quorumServers;

    public RecoverableZooKeeper(String quorumServers, int sessionTimeout, Watcher watcher, int maxRetries, int retryIntervalMillis) throws IOException {
        this.zk = new ZooKeeper(quorumServers, sessionTimeout, watcher);
        this.retryCounterFactory = new RetryCounterFactory(maxRetries, retryIntervalMillis);
        this.watcher = watcher;
        this.sessionTimeout = sessionTimeout;
        this.quorumServers = quorumServers;
    }

    private static synchronized List<String> filterByPrefix(List<String> nodes, String ... prefixes) {
        ArrayList<String> lockChildren = new ArrayList<String>();
        block0: for (String child : nodes) {
            for (String prefix : prefixes) {
                if (!child.startsWith(prefix)) continue;
                lockChildren.add(child);
                continue block0;
            }
        }
        return lockChildren;
    }

    public synchronized void reconnectAfterExpiration() throws IOException, InterruptedException {
        if (this.zk != null) {
            ZooKeeper.States state = this.zk.getState();
            if (state != null && state.isConnected() && state.isAlive()) {
                return;
            }
            logger.info("[ZK_SESSION_EXPIRATION] Closing dead ZooKeeper connection, session was: 0x" + Long.toHexString(this.zk.getSessionId()));
            this.zk.close();
        }
        this.zk = new ZooKeeper(this.quorumServers, this.sessionTimeout, this.watcher);
        logger.info("[ZK_SESSION_EXPIRATION] Recreated a ZooKeeper, session is: 0x" + Long.toHexString(this.zk.getSessionId()));
    }

    public synchronized void ensureConnectivity(KeeperException e) {
        if (e != null && e.code() != KeeperException.Code.SESSIONEXPIRED) {
            return;
        }
        try {
            this.reconnectAfterExpiration();
        }
        catch (Throwable e1) {
            logger.error("[ZK_SESSION_EXPIRATION] reconnectAfterExpiration failed.", e1);
        }
    }

    public synchronized void delete(String path, int version) throws InterruptedException, KeeperException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        boolean isRetry = false;
        while (true) {
            try {
                this.zk.delete(path, version);
                return;
            }
            catch (KeeperException e) {
                this.ensureConnectivity(e);
                switch (e.code()) {
                    case NONODE: {
                        if (isRetry) {
                            logger.info("Node " + path + " already deleted. Assuming that a previous attempt succeeded.");
                            return;
                        }
                        logger.warn("Node " + path + " already deleted, and this is not a retry");
                        throw e;
                    }
                    case CONNECTIONLOSS: 
                    case SESSIONEXPIRED: 
                    case OPERATIONTIMEOUT: {
                        this.retryOrThrow(retryCounter, e, "delete");
                        break;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                isRetry = true;
                continue;
            }
            break;
        }
    }

    public synchronized Stat exists(String path, Watcher watcher) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        while (true) {
            try {
                return this.zk.exists(path, watcher);
            }
            catch (KeeperException e) {
                this.ensureConnectivity(e);
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case SESSIONEXPIRED: 
                    case OPERATIONTIMEOUT: {
                        this.retryOrThrow(retryCounter, e, "exists");
                        break;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public synchronized Stat exists(String path, boolean watch) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        while (true) {
            try {
                return this.zk.exists(path, watch);
            }
            catch (KeeperException e) {
                this.ensureConnectivity(e);
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case SESSIONEXPIRED: 
                    case OPERATIONTIMEOUT: {
                        this.retryOrThrow(retryCounter, e, "exists");
                        break;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    private synchronized void retryOrThrow(RetryCounter retryCounter, KeeperException e, String opName) throws KeeperException {
        logger.warn("Possibly transient ZooKeeper exception: ", (Throwable)e);
        if (!retryCounter.shouldRetry()) {
            logger.error("ZooKeeper " + opName + " failed after " + retryCounter.getMaxRetries() + " retries");
            throw e;
        }
    }

    public synchronized List<String> getChildren(String path, Watcher watcher) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        while (true) {
            try {
                return this.zk.getChildren(path, watcher);
            }
            catch (KeeperException e) {
                this.ensureConnectivity(e);
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case SESSIONEXPIRED: 
                    case OPERATIONTIMEOUT: {
                        this.retryOrThrow(retryCounter, e, "getClientLst");
                        break;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public synchronized List<String> getChildren(String path, boolean watch) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        while (true) {
            try {
                return this.zk.getChildren(path, watch);
            }
            catch (KeeperException e) {
                this.ensureConnectivity(e);
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case SESSIONEXPIRED: 
                    case OPERATIONTIMEOUT: {
                        this.retryOrThrow(retryCounter, e, "getClientLst");
                        break;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public synchronized byte[] getData(String path, Watcher watcher, Stat stat) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        while (true) {
            try {
                byte[] revData = this.zk.getData(path, watcher, stat);
                return revData;
            }
            catch (KeeperException e) {
                this.ensureConnectivity(e);
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case SESSIONEXPIRED: 
                    case OPERATIONTIMEOUT: {
                        this.retryOrThrow(retryCounter, e, "getData");
                        break;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public synchronized byte[] getData(String path, boolean watch, Stat stat) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        while (true) {
            try {
                byte[] revData = this.zk.getData(path, watch, stat);
                return revData;
            }
            catch (KeeperException e) {
                this.ensureConnectivity(e);
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case SESSIONEXPIRED: 
                    case OPERATIONTIMEOUT: {
                        this.retryOrThrow(retryCounter, e, "getData");
                        break;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public synchronized Stat setData(String path, byte[] data, int version) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        byte[] newData = data;
        while (true) {
            try {
                return this.zk.setData(path, newData, version);
            }
            catch (KeeperException e) {
                this.ensureConnectivity(e);
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case SESSIONEXPIRED: 
                    case OPERATIONTIMEOUT: {
                        this.retryOrThrow(retryCounter, e, "setData");
                        break;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    public synchronized String create(String path, byte[] data, List<ACL> acl, CreateMode createMode) throws KeeperException, InterruptedException {
        byte[] newData = data;
        switch (createMode) {
            case EPHEMERAL: 
            case PERSISTENT: {
                return this.createNonSequential(path, newData, acl, createMode);
            }
            case EPHEMERAL_SEQUENTIAL: 
            case PERSISTENT_SEQUENTIAL: {
                return this.createSequential(path, newData, acl, createMode);
            }
        }
        throw new IllegalArgumentException("Unrecognized CreateMode: " + createMode);
    }

    private synchronized String createNonSequential(String path, byte[] data, List<ACL> acl, CreateMode createMode) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        boolean isRetry = false;
        while (true) {
            try {
                return this.zk.create(path, data, acl, createMode);
            }
            catch (KeeperException e) {
                this.ensureConnectivity(e);
                switch (e.code()) {
                    case NODEEXISTS: {
                        if (isRetry) {
                            byte[] currentData = this.zk.getData(path, false, null);
                            if (currentData != null && Bytes.compareTo(currentData, data) == 0) {
                                return path;
                            }
                            logger.error("Node " + path + " already exists with " + Bytes.toStringBinary(currentData) + ", could not write " + Bytes.toStringBinary(data));
                            throw e;
                        }
                        throw e;
                    }
                    case CONNECTIONLOSS: 
                    case SESSIONEXPIRED: 
                    case OPERATIONTIMEOUT: {
                        this.retryOrThrow(retryCounter, e, "create");
                        break;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                isRetry = true;
                continue;
            }
            break;
        }
    }

    private synchronized String createSequential(String path, byte[] data, List<ACL> acl, CreateMode createMode) throws KeeperException, InterruptedException {
        RetryCounter retryCounter = this.retryCounterFactory.create();
        boolean first = true;
        String newPath = path;
        while (true) {
            try {
                String previousResult;
                if (!first && (previousResult = this.findPreviousSequentialNode(newPath)) != null) {
                    return previousResult;
                }
                first = false;
                return this.zk.create(newPath, data, acl, createMode);
            }
            catch (KeeperException e) {
                this.ensureConnectivity(e);
                switch (e.code()) {
                    case CONNECTIONLOSS: 
                    case SESSIONEXPIRED: 
                    case OPERATIONTIMEOUT: {
                        this.retryOrThrow(retryCounter, e, "create");
                        break;
                    }
                    default: {
                        throw e;
                    }
                }
                retryCounter.sleepUntilNextRetry();
                retryCounter.useRetry();
                continue;
            }
            break;
        }
    }

    private synchronized String findPreviousSequentialNode(String path) throws KeeperException, InterruptedException {
        int lastSlashIdx = path.lastIndexOf(47);
        assert (lastSlashIdx != -1);
        String parent = path.substring(0, lastSlashIdx);
        String nodePrefix = path.substring(lastSlashIdx + 1);
        List nodes = this.zk.getChildren(parent, false);
        List<String> matching = RecoverableZooKeeper.filterByPrefix(nodes, nodePrefix);
        for (String node : matching) {
            String nodePath = parent + "/" + node;
            Stat stat = this.zk.exists(nodePath, false);
            if (stat == null) continue;
            return nodePath;
        }
        return null;
    }

    public synchronized long getSessionId() {
        return this.zk.getSessionId();
    }

    public synchronized void close() throws InterruptedException {
        if (this.zk != null) {
            this.zk.close();
        }
    }

    public synchronized ZooKeeper.States getState() {
        return this.zk.getState();
    }

    public synchronized ZooKeeper getZooKeeper() {
        return this.zk;
    }

    public synchronized byte[] getSessionPasswd() {
        return this.zk.getSessionPasswd();
    }

    public synchronized void sync(String path, AsyncCallback.VoidCallback cb, Object ctx) {
        this.zk.sync(path, null, null);
    }
}

