/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ozone.rocksdiff;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.graph.MutableGraph;
import com.google.protobuf.InvalidProtocolBufferException;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.hdds.StringUtils;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.utils.IOUtils;
import org.apache.hadoop.hdds.utils.NativeLibraryNotLoadedException;
import org.apache.hadoop.hdds.utils.Scheduler;
import org.apache.hadoop.hdds.utils.db.RocksDatabaseException;
import org.apache.hadoop.hdds.utils.db.managed.ManagedDBOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedEnvOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedRawSSTFileIterator;
import org.apache.hadoop.hdds.utils.db.managed.ManagedRawSSTFileReader;
import org.apache.hadoop.hdds.utils.db.managed.ManagedRocksDB;
import org.apache.hadoop.hdds.utils.db.managed.ManagedRocksIterator;
import org.apache.hadoop.hdds.utils.db.managed.ManagedSstFileWriter;
import org.apache.hadoop.ozone.OzoneConfigKeys;
import org.apache.hadoop.ozone.lock.BootstrapStateHandler;
import org.apache.ozone.compaction.log.CompactionFileInfo;
import org.apache.ozone.compaction.log.CompactionLogEntry;
import org.apache.ozone.rocksdb.util.RdbUtil;
import org.apache.ozone.rocksdiff.CompactionDag;
import org.apache.ozone.rocksdiff.CompactionNode;
import org.apache.ozone.rocksdiff.DifferSnapshotInfo;
import org.apache.ozone.rocksdiff.RocksDiffUtils;
import org.apache.ozone.rocksdiff.SSTFilePruningMetrics;
import org.rocksdb.AbstractEventListener;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.CompactionJobInfo;
import org.rocksdb.EnvOptions;
import org.rocksdb.LiveFileMetaData;
import org.rocksdb.Options;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RocksDBCheckpointDiffer
implements AutoCloseable,
BootstrapStateHandler {
    private static final Logger LOG = LoggerFactory.getLogger(RocksDBCheckpointDiffer.class);
    private final String metadataDir;
    private final String sstBackupDir;
    private final String compactionLogDir;
    public static final String COMPACTION_LOG_FILE_NAME_SUFFIX = ".log";
    private static final String COMPACTION_LOG_COMMENT_LINE_PREFIX = "# ";
    private static final String COMPACTION_LOG_ENTRY_LINE_PREFIX = "C ";
    private static final String COMPACTION_LOG_SEQ_NUM_LINE_PREFIX = "S ";
    private static final String COMPACTION_LOG_ENTRY_FILE_DELIMITER = ",";
    private static final String SPACE_DELIMITER = " ";
    private static final String COMPACTION_LOG_ENTRY_INPUT_OUTPUT_FILES_DELIMITER = ":";
    static final String SST_FILE_EXTENSION = ".sst";
    public static final int SST_FILE_EXTENSION_LENGTH = ".sst".length();
    static final String PRUNED_SST_FILE_TEMP = "pruned.sst.tmp";
    private static final int LONG_MAX_STR_LEN = String.valueOf(Long.MAX_VALUE).length();
    private long reconstructionSnapshotCreationTime;
    private String reconstructionCompactionReason;
    private final Scheduler scheduler;
    private volatile boolean closed;
    private final long maxAllowedTimeInDag;
    private final BootstrapStateHandler.Lock lock = new BootstrapStateHandler.Lock();
    private static final int SST_READ_AHEAD_SIZE = 0x200000;
    private int pruneSSTFileBatchSize;
    private SSTFilePruningMetrics sstFilePruningMetrics;
    private ColumnFamilyHandle snapshotInfoTableCFHandle;
    private static final String DAG_PRUNING_SERVICE_NAME = "CompactionDagPruningService";
    private AtomicBoolean suspended;
    private ColumnFamilyHandle compactionLogTableCFHandle;
    private ManagedRocksDB activeRocksDB;
    private final ConcurrentMap<String, CompactionFileInfo> inflightCompactions;
    private Queue<byte[]> pruneQueue = null;
    public static final Set<String> COLUMN_FAMILIES_TO_TRACK_IN_DAG = ImmutableSet.of((Object)"keyTable", (Object)"directoryTable", (Object)"fileTable");
    private final CompactionDag compactionDag;

    @VisibleForTesting
    RocksDBCheckpointDiffer(String metadataDirName, String sstBackupDirName, String compactionLogDirName, String activeDBLocationName, ConfigurationSource configuration) {
        Preconditions.checkNotNull((Object)metadataDirName);
        Preconditions.checkNotNull((Object)sstBackupDirName);
        Preconditions.checkNotNull((Object)compactionLogDirName);
        Preconditions.checkNotNull((Object)activeDBLocationName);
        this.metadataDir = metadataDirName;
        this.compactionLogDir = this.createCompactionLogDir(metadataDirName, compactionLogDirName);
        this.sstBackupDir = Paths.get(metadataDirName, sstBackupDirName) + "/";
        this.createSstBackUpDir();
        this.maxAllowedTimeInDag = configuration.getTimeDuration("ozone.om.snapshot.compaction.dag.max.time.allowed", OzoneConfigKeys.OZONE_OM_SNAPSHOT_COMPACTION_DAG_MAX_TIME_ALLOWED_DEFAULT, TimeUnit.MILLISECONDS);
        this.suspended = new AtomicBoolean(false);
        long pruneCompactionDagDaemonRunIntervalInMs = configuration.getTimeDuration("ozone.om.snapshot.compaction.dag.prune.daemon.run.interval", OzoneConfigKeys.OZONE_OM_SNAPSHOT_PRUNE_COMPACTION_DAG_DAEMON_RUN_INTERVAL_DEFAULT, TimeUnit.MILLISECONDS);
        this.pruneSSTFileBatchSize = configuration.getInt("ozone.om.snapshot.prune.compaction.backup.batch.size", 2000);
        this.sstFilePruningMetrics = SSTFilePruningMetrics.create(activeDBLocationName);
        try {
            if (configuration.getBoolean("ozone.om.snapshot.load.native.lib", true) && ManagedRawSSTFileReader.loadLibrary()) {
                this.pruneQueue = new ConcurrentLinkedQueue<byte[]>();
            }
        }
        catch (NativeLibraryNotLoadedException e) {
            LOG.warn("Native Library for raw sst file reading loading failed. Cannot prune OMKeyInfo from SST files. {}", (Object)e.getMessage());
        }
        if (pruneCompactionDagDaemonRunIntervalInMs > 0L) {
            this.scheduler = new Scheduler(DAG_PRUNING_SERVICE_NAME, true, 1);
            this.scheduler.scheduleWithFixedDelay(this::pruneOlderSnapshotsWithCompactionHistory, pruneCompactionDagDaemonRunIntervalInMs, pruneCompactionDagDaemonRunIntervalInMs, TimeUnit.MILLISECONDS);
            this.scheduler.scheduleWithFixedDelay(this::pruneSstFiles, pruneCompactionDagDaemonRunIntervalInMs, pruneCompactionDagDaemonRunIntervalInMs, TimeUnit.MILLISECONDS);
            if (this.pruneQueue != null) {
                this.scheduler.scheduleWithFixedDelay(this::pruneSstFileValues, pruneCompactionDagDaemonRunIntervalInMs, pruneCompactionDagDaemonRunIntervalInMs, TimeUnit.MILLISECONDS);
            }
        } else {
            this.scheduler = null;
        }
        this.inflightCompactions = new ConcurrentHashMap<String, CompactionFileInfo>();
        this.compactionDag = new CompactionDag();
    }

    private String createCompactionLogDir(String metadataDirName, String compactionLogDirName) {
        File parentDir = new File(metadataDirName);
        if (!parentDir.exists() && !parentDir.mkdirs()) {
            LOG.error("Error creating compaction log parent dir.");
            return null;
        }
        String compactionLogDirectory = Paths.get(metadataDirName, compactionLogDirName).toString();
        File clDir = new File(compactionLogDirectory);
        if (!clDir.exists() && !clDir.mkdir()) {
            LOG.error("Error creating compaction log dir.");
            return null;
        }
        Path readmePath = Paths.get(compactionLogDirectory, "_README.txt");
        File readmeFile = new File(readmePath.toString());
        if (!readmeFile.exists()) {
            try (BufferedWriter bw = Files.newBufferedWriter(readmePath, StandardOpenOption.CREATE);){
                bw.write("This directory holds Ozone Manager RocksDB compaction logs.\nDO NOT add, change or delete any files in this directory unless you know what you are doing.\n");
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return compactionLogDirectory + "/";
    }

    private void createSstBackUpDir() {
        File dir = new File(this.sstBackupDir);
        if (!dir.exists() && !dir.mkdir()) {
            String errorMsg = "Failed to create SST file backup directory. Check if OM has write permission.";
            LOG.error(errorMsg);
            throw new RuntimeException(errorMsg);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        if (!this.closed) {
            RocksDBCheckpointDiffer rocksDBCheckpointDiffer = this;
            synchronized (rocksDBCheckpointDiffer) {
                if (!this.closed) {
                    this.closed = true;
                    if (this.scheduler != null) {
                        LOG.info("Shutting down {}.", (Object)DAG_PRUNING_SERVICE_NAME);
                        this.scheduler.close();
                    }
                    if (this.sstFilePruningMetrics != null) {
                        this.sstFilePruningMetrics.unRegister();
                    }
                }
            }
        }
    }

    public void setRocksDBForCompactionTracking(ManagedDBOptions rocksOptions) {
        ArrayList<AbstractEventListener> events = new ArrayList<AbstractEventListener>();
        events.add(this.newCompactionBeginListener());
        events.add(this.newCompactionCompletedListener());
        rocksOptions.setListeners(events);
    }

    public void setSnapshotInfoTableCFHandle(ColumnFamilyHandle snapshotInfoTableCFHandle) {
        Preconditions.checkNotNull((Object)snapshotInfoTableCFHandle, (Object)"Column family handle should not be null");
        this.snapshotInfoTableCFHandle = snapshotInfoTableCFHandle;
    }

    public synchronized void setCompactionLogTableCFHandle(ColumnFamilyHandle compactionLogTableCFHandle) {
        Preconditions.checkNotNull((Object)compactionLogTableCFHandle, (Object)"Column family handle should not be null");
        this.compactionLogTableCFHandle = compactionLogTableCFHandle;
    }

    public synchronized void setActiveRocksDB(ManagedRocksDB activeRocksDB) {
        Preconditions.checkNotNull((Object)activeRocksDB, (Object)"RocksDB should not be null.");
        this.activeRocksDB = activeRocksDB;
    }

    private boolean isSnapshotInfoTableEmpty(RocksDB db) {
        if (this.snapshotInfoTableCFHandle == null) {
            LOG.warn("Snapshot info table column family handle is not set!");
            return false;
        }
        try (ManagedRocksIterator it = ManagedRocksIterator.managed((RocksIterator)db.newIterator(this.snapshotInfoTableCFHandle));){
            ((RocksIterator)it.get()).seekToFirst();
            boolean bl = !((RocksIterator)it.get()).isValid();
            return bl;
        }
    }

    @VisibleForTesting
    boolean shouldSkipCompaction(byte[] columnFamilyBytes, List<String> inputFiles, List<String> outputFiles) {
        String columnFamily = StringUtils.bytes2String((byte[])columnFamilyBytes);
        if (!COLUMN_FAMILIES_TO_TRACK_IN_DAG.contains(columnFamily)) {
            LOG.debug("Skipping compaction for columnFamily: {}", (Object)columnFamily);
            return true;
        }
        if (inputFiles.isEmpty()) {
            LOG.debug("Compaction input files list is empty");
            return true;
        }
        if (new HashSet<String>(inputFiles).equals(new HashSet<String>(outputFiles))) {
            LOG.info("Skipped the compaction entry. Compaction input files: {} and output files: {} are same.", inputFiles, outputFiles);
            return true;
        }
        return false;
    }

    private AbstractEventListener newCompactionBeginListener() {
        return new AbstractEventListener(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void onCompactionBegin(RocksDB db, CompactionJobInfo compactionJobInfo) {
                if (RocksDBCheckpointDiffer.this.shouldSkipCompaction(compactionJobInfo.columnFamilyName(), compactionJobInfo.inputFiles(), compactionJobInfo.outputFiles())) {
                    return;
                }
                1 var3_3 = this;
                synchronized (var3_3) {
                    if (RocksDBCheckpointDiffer.this.closed) {
                        return;
                    }
                    if (RocksDBCheckpointDiffer.this.isSnapshotInfoTableEmpty(db)) {
                        return;
                    }
                }
                RocksDBCheckpointDiffer.this.inflightCompactions.putAll(RocksDBCheckpointDiffer.this.toFileInfoList(compactionJobInfo.inputFiles(), db));
                for (String file : compactionJobInfo.inputFiles()) {
                    RocksDBCheckpointDiffer.this.createLink(Paths.get(RocksDBCheckpointDiffer.this.sstBackupDir, new File(file).getName()), Paths.get(file, new String[0]));
                }
            }
        };
    }

    private AbstractEventListener newCompactionCompletedListener() {
        return new AbstractEventListener(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void onCompactionCompleted(RocksDB db, CompactionJobInfo compactionJobInfo) {
                byte[] key;
                if (RocksDBCheckpointDiffer.this.shouldSkipCompaction(compactionJobInfo.columnFamilyName(), compactionJobInfo.inputFiles(), compactionJobInfo.outputFiles())) {
                    return;
                }
                long trxId = db.getLatestSequenceNumber();
                Map inputFileCompactions = RocksDBCheckpointDiffer.this.toFileInfoList(compactionJobInfo.inputFiles(), db);
                CompactionLogEntry.Builder builder = new CompactionLogEntry.Builder(trxId, System.currentTimeMillis(), inputFileCompactions.keySet().stream().map(inputFile -> {
                    if (!RocksDBCheckpointDiffer.this.inflightCompactions.containsKey(inputFile)) {
                        LOG.warn("Input file not found in inflightCompactionsMap : {} which should have been added on compactionBeginListener.", inputFile);
                    }
                    return RocksDBCheckpointDiffer.this.inflightCompactions.getOrDefault(inputFile, (CompactionFileInfo)inputFileCompactions.get(inputFile));
                }).collect(Collectors.toList()), new ArrayList<CompactionFileInfo>(RocksDBCheckpointDiffer.this.toFileInfoList(compactionJobInfo.outputFiles(), db).values()));
                if (LOG.isDebugEnabled()) {
                    builder = builder.setCompactionReason(compactionJobInfo.compactionReason().toString());
                }
                CompactionLogEntry compactionLogEntry = builder.build();
                2 var9_7 = this;
                synchronized (var9_7) {
                    if (RocksDBCheckpointDiffer.this.closed) {
                        return;
                    }
                    if (RocksDBCheckpointDiffer.this.isSnapshotInfoTableEmpty(db)) {
                        return;
                    }
                    key = RocksDBCheckpointDiffer.this.addToCompactionLogTable(compactionLogEntry);
                    RocksDBCheckpointDiffer.this.compactionDag.populateCompactionDAG(compactionLogEntry.getInputFileInfoList(), compactionLogEntry.getOutputFileInfoList(), compactionLogEntry.getDbSequenceNumber());
                    for (String inputFile2 : inputFileCompactions.keySet()) {
                        RocksDBCheckpointDiffer.this.inflightCompactions.remove(inputFile2);
                    }
                }
                if (RocksDBCheckpointDiffer.this.pruneQueue != null) {
                    RocksDBCheckpointDiffer.this.pruneQueue.offer(key);
                    RocksDBCheckpointDiffer.this.sstFilePruningMetrics.updateQueueSize(RocksDBCheckpointDiffer.this.pruneQueue.size());
                }
            }
        };
    }

    @VisibleForTesting
    byte[] addToCompactionLogTable(CompactionLogEntry compactionLogEntry) {
        String dbSequenceIdStr = String.valueOf(compactionLogEntry.getDbSequenceNumber());
        if (dbSequenceIdStr.length() < LONG_MAX_STR_LEN) {
            dbSequenceIdStr = org.apache.commons.lang3.StringUtils.leftPad((String)dbSequenceIdStr, (int)LONG_MAX_STR_LEN, (String)"0");
        }
        String keyString = dbSequenceIdStr + "-" + compactionLogEntry.getCompactionTime();
        byte[] key = keyString.getBytes(StandardCharsets.UTF_8);
        byte[] value = compactionLogEntry.getProtobuf().toByteArray();
        try {
            ((RocksDB)this.activeRocksDB.get()).put(this.compactionLogTableCFHandle, key, value);
        }
        catch (RocksDBException exception) {
            throw new RuntimeException(exception);
        }
        return key;
    }

    private void createLink(Path link, Path source) {
        try {
            Files.createLink(link, source);
        }
        catch (FileAlreadyExistsException ignored) {
            LOG.debug("SST file already exists: {}", (Object)source);
        }
        catch (IOException e) {
            LOG.error("Exception in creating hard link for {}", (Object)source);
            throw new RuntimeException("Failed to create hard link", e);
        }
    }

    private String trimSSTFilename(String filename) {
        if (!filename.startsWith("/")) {
            String errorMsg = String.format("Invalid start of filename: '%s'. Expected '/'", filename);
            LOG.error(errorMsg);
            throw new RuntimeException(errorMsg);
        }
        if (!filename.endsWith(SST_FILE_EXTENSION)) {
            String errorMsg = String.format("Invalid extension of file: '%s'. Expected '%s'", filename, SST_FILE_EXTENSION_LENGTH);
            LOG.error(errorMsg);
            throw new RuntimeException(errorMsg);
        }
        return filename.substring("/".length(), filename.length() - SST_FILE_EXTENSION_LENGTH);
    }

    public Set<String> readRocksDBLiveFiles(ManagedRocksDB rocksDB) {
        HashSet<String> liveFiles = new HashSet<String>();
        List<String> cfs = Arrays.asList(StringUtils.bytes2String((byte[])RocksDB.DEFAULT_COLUMN_FAMILY), "keyTable", "directoryTable", "fileTable");
        List<LiveFileMetaData> liveFileMetaDataList = RdbUtil.getLiveSSTFilesForCFs(rocksDB, cfs);
        LOG.debug("SST File Metadata for DB: " + ((RocksDB)rocksDB.get()).getName());
        for (LiveFileMetaData m : liveFileMetaDataList) {
            LOG.debug("File: {}, Level: {}", (Object)m.fileName(), (Object)m.level());
            String trimmedFilename = this.trimSSTFilename(m.fileName());
            liveFiles.add(trimmedFilename);
        }
        return liveFiles;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void processCompactionLogLine(String line) {
        LOG.debug("Processing line: {}", (Object)line);
        RocksDBCheckpointDiffer rocksDBCheckpointDiffer = this;
        synchronized (rocksDBCheckpointDiffer) {
            if (line.startsWith(COMPACTION_LOG_COMMENT_LINE_PREFIX)) {
                this.reconstructionCompactionReason = line.substring(COMPACTION_LOG_COMMENT_LINE_PREFIX.length());
            } else if (line.startsWith(COMPACTION_LOG_SEQ_NUM_LINE_PREFIX)) {
                this.reconstructionSnapshotCreationTime = this.getSnapshotCreationTimeFromLogLine(line);
            } else if (line.startsWith(COMPACTION_LOG_ENTRY_LINE_PREFIX)) {
                String[] lineSpilt = line.split(SPACE_DELIMITER);
                if (lineSpilt.length != 3) {
                    LOG.error("Invalid line in compaction log: {}", (Object)line);
                    return;
                }
                String dbSequenceNumber = lineSpilt[1];
                String[] io = lineSpilt[2].split(COMPACTION_LOG_ENTRY_INPUT_OUTPUT_FILES_DELIMITER);
                if (io.length != 2) {
                    if (line.endsWith(COMPACTION_LOG_ENTRY_INPUT_OUTPUT_FILES_DELIMITER)) {
                        LOG.debug("Ignoring compaction log line for SST deletion");
                    } else {
                        LOG.error("Invalid line in compaction log: {}", (Object)line);
                    }
                    return;
                }
                String[] inputFiles = io[0].split(COMPACTION_LOG_ENTRY_FILE_DELIMITER);
                String[] outputFiles = io[1].split(COMPACTION_LOG_ENTRY_FILE_DELIMITER);
                this.addFileInfoToCompactionLogTable(Long.parseLong(dbSequenceNumber), this.reconstructionSnapshotCreationTime, inputFiles, outputFiles, this.reconstructionCompactionReason);
            } else {
                LOG.error("Invalid line in compaction log: {}", (Object)line);
            }
        }
    }

    private void readCompactionLogFile(String currCompactionLogPath) {
        LOG.debug("Loading compaction log: {}", (Object)currCompactionLogPath);
        try (Stream<String> logLineStream = Files.lines(Paths.get(currCompactionLogPath, new String[0]), StandardCharsets.UTF_8);){
            logLineStream.forEach(this::processCompactionLogLine);
        }
        catch (IOException ioEx) {
            throw new RuntimeException(ioEx);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addEntriesFromLogFilesToDagAndCompactionLogTable() {
        RocksDBCheckpointDiffer rocksDBCheckpointDiffer = this;
        synchronized (rocksDBCheckpointDiffer) {
            this.reconstructionSnapshotCreationTime = 0L;
            this.reconstructionCompactionReason = null;
            try (Stream<Path> pathStream = Files.list(Paths.get(this.compactionLogDir, new String[0])).filter(e -> e.toString().toLowerCase().endsWith(COMPACTION_LOG_FILE_NAME_SUFFIX)).sorted();){
                for (Path logPath : pathStream.collect(Collectors.toList())) {
                    this.readCompactionLogFile(logPath.toString());
                    Files.delete(logPath);
                }
            }
            catch (IOException e2) {
                throw new RuntimeException("Error listing compaction log dir " + this.compactionLogDir, e2);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void loadAllCompactionLogs() {
        RocksDBCheckpointDiffer rocksDBCheckpointDiffer = this;
        synchronized (rocksDBCheckpointDiffer) {
            this.preconditionChecksForLoadAllCompactionLogs();
            this.addEntriesFromLogFilesToDagAndCompactionLogTable();
            this.loadCompactionDagFromDB();
        }
    }

    private void loadCompactionDagFromDB() {
        try (ManagedRocksIterator managedRocksIterator = new ManagedRocksIterator(((RocksDB)this.activeRocksDB.get()).newIterator(this.compactionLogTableCFHandle));){
            ((RocksIterator)managedRocksIterator.get()).seekToFirst();
            while (((RocksIterator)managedRocksIterator.get()).isValid()) {
                byte[] value = ((RocksIterator)managedRocksIterator.get()).value();
                CompactionLogEntry compactionLogEntry = CompactionLogEntry.getFromProtobuf(HddsProtos.CompactionLogEntryProto.parseFrom((byte[])value));
                this.compactionDag.populateCompactionDAG(compactionLogEntry.getInputFileInfoList(), compactionLogEntry.getOutputFileInfoList(), compactionLogEntry.getDbSequenceNumber());
                if (this.pruneQueue != null) {
                    this.pruneQueue.offer(((RocksIterator)managedRocksIterator.get()).key());
                }
                ((RocksIterator)managedRocksIterator.get()).next();
            }
        }
        catch (InvalidProtocolBufferException e) {
            throw new RuntimeException(e);
        }
        finally {
            if (this.pruneQueue != null) {
                this.sstFilePruningMetrics.updateQueueSize(this.pruneQueue.size());
            }
        }
    }

    private void preconditionChecksForLoadAllCompactionLogs() {
        Preconditions.checkNotNull((Object)this.compactionLogDir, (Object)"Compaction log directory must be set.");
        Preconditions.checkNotNull((Object)this.compactionLogTableCFHandle, (Object)"compactionLogTableCFHandle must be set before calling loadAllCompactionLogs.");
        Preconditions.checkNotNull((Object)this.activeRocksDB, (Object)"activeRocksDB must be set before calling loadAllCompactionLogs.");
    }

    private String getSSTFullPath(String sstFilenameWithoutExtension, String ... dbPaths) {
        Path sstPathInBackupDir = Paths.get(this.sstBackupDir, sstFilenameWithoutExtension + SST_FILE_EXTENSION);
        if (Files.exists(sstPathInBackupDir, new LinkOption[0])) {
            return sstPathInBackupDir.toString();
        }
        for (String dbPath : dbPaths) {
            Path sstPathInDBDir = Paths.get(dbPath, sstFilenameWithoutExtension + SST_FILE_EXTENSION);
            if (!Files.exists(sstPathInDBDir, new LinkOption[0])) continue;
            return sstPathInDBDir.toString();
        }
        throw new RuntimeException("Unable to locate SST file: " + sstFilenameWithoutExtension);
    }

    public synchronized Optional<List<String>> getSSTDiffListWithFullPath(DifferSnapshotInfo src, DifferSnapshotInfo dest, String sstFilesDirForSnapDiffJob) {
        Optional<List<String>> sstDiffList = this.getSSTDiffList(src, dest);
        return sstDiffList.map(diffList -> diffList.stream().map(sst -> {
            String sstFullPath = this.getSSTFullPath((String)sst, src.getDbPath(), dest.getDbPath());
            Path link = Paths.get(sstFilesDirForSnapDiffJob, sst + SST_FILE_EXTENSION);
            Path srcFile = Paths.get(sstFullPath, new String[0]);
            this.createLink(link, srcFile);
            return link.toString();
        }).collect(Collectors.toList()));
    }

    public synchronized Optional<List<String>> getSSTDiffList(DifferSnapshotInfo src, DifferSnapshotInfo dest) {
        Set<String> srcSnapFiles = this.readRocksDBLiveFiles(src.getRocksDB());
        Set<String> destSnapFiles = this.readRocksDBLiveFiles(dest.getRocksDB());
        HashSet<String> fwdDAGSameFiles = new HashSet<String>();
        HashSet<String> fwdDAGDifferentFiles = new HashSet<String>();
        LOG.debug("Doing forward diff from src '{}' to dest '{}'", (Object)src.getDbPath(), (Object)dest.getDbPath());
        this.internalGetSSTDiffList(src, dest, srcSnapFiles, destSnapFiles, fwdDAGSameFiles, fwdDAGDifferentFiles);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Result of diff from src '" + src.getDbPath() + "' to dest '" + dest.getDbPath() + "':");
            StringBuilder logSB = new StringBuilder();
            logSB.append("Fwd DAG same SST files:      ");
            for (String file : fwdDAGSameFiles) {
                logSB.append(file).append(SPACE_DELIMITER);
            }
            LOG.debug(logSB.toString());
            logSB.setLength(0);
            logSB.append("Fwd DAG different SST files: ");
            for (String file : fwdDAGDifferentFiles) {
                logSB.append(file).append(SPACE_DELIMITER);
            }
            LOG.debug("{}", (Object)logSB);
        }
        for (String destSnapFile : destSnapFiles) {
            if (fwdDAGSameFiles.contains(destSnapFile) || fwdDAGDifferentFiles.contains(destSnapFile)) continue;
            return Optional.empty();
        }
        if (src.getTablePrefixes() != null && !src.getTablePrefixes().isEmpty()) {
            RocksDiffUtils.filterRelevantSstFiles(fwdDAGDifferentFiles, src.getTablePrefixes(), this.compactionDag.getCompactionMap(), src.getRocksDB(), dest.getRocksDB());
        }
        return Optional.of(new ArrayList<String>(fwdDAGDifferentFiles));
    }

    synchronized void internalGetSSTDiffList(DifferSnapshotInfo src, DifferSnapshotInfo dest, Set<String> srcSnapFiles, Set<String> destSnapFiles, Set<String> sameFiles, Set<String> differentFiles) {
        Preconditions.checkArgument((boolean)sameFiles.isEmpty(), (Object)"Set must be empty");
        Preconditions.checkArgument((boolean)differentFiles.isEmpty(), (Object)"Set must be empty");
        for (String fileName : srcSnapFiles) {
            if (destSnapFiles.contains(fileName)) {
                LOG.debug("Source '{}' and destination '{}' share the same SST '{}'", new Object[]{src.getDbPath(), dest.getDbPath(), fileName});
                sameFiles.add(fileName);
                continue;
            }
            CompactionNode infileNode = this.compactionDag.getCompactionNode(fileName);
            if (infileNode == null) {
                LOG.debug("Source '{}' SST file '{}' is never compacted", (Object)src.getDbPath(), (Object)fileName);
                differentFiles.add(fileName);
                continue;
            }
            LOG.debug("Expanding SST file: {}", (Object)fileName);
            HashSet<CompactionNode> currentLevel = new HashSet<CompactionNode>();
            currentLevel.add(infileNode);
            int level = 1;
            while (!currentLevel.isEmpty()) {
                LOG.debug("Traversal level: {}. Current level has {} nodes.", (Object)level++, (Object)currentLevel.size());
                if (level >= 1000000) {
                    String errorMsg = String.format("Graph traversal level exceeded allowed maximum (%d). This could be due to invalid input generating a loop in the traversal path. Same SSTs found so far: %s, different SSTs: %s", level, sameFiles, differentFiles);
                    LOG.error(errorMsg);
                    sameFiles.clear();
                    differentFiles.clear();
                    throw new RuntimeException(errorMsg);
                }
                HashSet<CompactionNode> nextLevel = new HashSet<CompactionNode>();
                for (CompactionNode current : currentLevel) {
                    LOG.debug("Processing node: '{}'", (Object)current.getFileName());
                    if (current.getSnapshotGeneration() < dest.getSnapshotGeneration()) {
                        LOG.debug("Current node's snapshot generation '{}' reached destination snapshot's '{}'. Src '{}' and dest '{}' have different SST file: '{}'", new Object[]{current.getSnapshotGeneration(), dest.getSnapshotGeneration(), src.getDbPath(), dest.getDbPath(), current.getFileName()});
                        differentFiles.add(current.getFileName());
                        continue;
                    }
                    Set successors = this.compactionDag.getForwardCompactionDAG().successors((Object)current);
                    if (successors.isEmpty()) {
                        LOG.debug("No further compaction happened to the current file. Src '{}' and dest '{}' have different file: {}", new Object[]{src.getDbPath(), dest.getDbPath(), current.getFileName()});
                        differentFiles.add(current.getFileName());
                        continue;
                    }
                    for (CompactionNode nextNode : successors) {
                        if (sameFiles.contains(nextNode.getFileName()) || differentFiles.contains(nextNode.getFileName())) {
                            LOG.debug("Skipping known processed SST: {}", (Object)nextNode.getFileName());
                            continue;
                        }
                        if (destSnapFiles.contains(nextNode.getFileName())) {
                            LOG.debug("Src '{}' and dest '{}' have the same SST: {}", new Object[]{src.getDbPath(), dest.getDbPath(), nextNode.getFileName()});
                            sameFiles.add(nextNode.getFileName());
                            continue;
                        }
                        LOG.debug("Src '{}' and dest '{}' have a different SST: {}", new Object[]{src.getDbPath(), dest.getDbPath(), nextNode.getFileName()});
                        nextLevel.add(nextNode);
                    }
                }
                currentLevel = nextLevel;
            }
        }
    }

    public String getMetadataDir() {
        return this.metadataDir;
    }

    @VisibleForTesting
    void dumpCompactionNodeTable() {
        List nodeList = this.compactionDag.getCompactionMap().values().stream().sorted(new NodeComparator()).collect(Collectors.toList());
        for (CompactionNode n : nodeList) {
            LOG.debug("File '{}' total keys: {}", (Object)n.getFileName(), (Object)n.getTotalNumberOfKeys());
            LOG.debug("File '{}' cumulative keys: {}", (Object)n.getFileName(), (Object)n.getCumulativeKeysReverseTraversal());
        }
    }

    @VisibleForTesting
    public MutableGraph<CompactionNode> getForwardCompactionDAG() {
        return this.compactionDag.getForwardCompactionDAG();
    }

    @VisibleForTesting
    public MutableGraph<CompactionNode> getBackwardCompactionDAG() {
        return this.compactionDag.getBackwardCompactionDAG();
    }

    private void addFileInfoToCompactionLogTable(long dbSequenceNumber, long creationTime, String[] inputFiles, String[] outputFiles, String compactionReason) {
        List<CompactionFileInfo> inputFileInfoList = Arrays.stream(inputFiles).map(inputFile -> new CompactionFileInfo.Builder((String)inputFile).build()).collect(Collectors.toList());
        List<CompactionFileInfo> outputFileInfoList = Arrays.stream(outputFiles).map(outputFile -> new CompactionFileInfo.Builder((String)outputFile).build()).collect(Collectors.toList());
        CompactionLogEntry.Builder builder = new CompactionLogEntry.Builder(dbSequenceNumber, creationTime, inputFileInfoList, outputFileInfoList);
        if (compactionReason != null) {
            builder.setCompactionReason(compactionReason);
        }
        this.addToCompactionLogTable(builder.build());
    }

    public void pruneOlderSnapshotsWithCompactionHistory() {
        if (!this.shouldRun()) {
            return;
        }
        Pair<Set<String>, List<byte[]>> fileNodeToKeyPair = this.getOlderFileNodes();
        Set lastCompactionSstFiles = (Set)fileNodeToKeyPair.getLeft();
        List keysToRemove = (List)fileNodeToKeyPair.getRight();
        Set<String> sstFileNodesRemoved = this.pruneSstFileNodesFromDag(lastCompactionSstFiles);
        if (CollectionUtils.isNotEmpty(sstFileNodesRemoved)) {
            LOG.info("Removing SST files: {} as part of compaction DAG pruning.", sstFileNodesRemoved);
        }
        try (BootstrapStateHandler.Lock lock = this.getBootstrapStateLock().lock();){
            this.removeSstFiles(sstFileNodesRemoved);
            this.removeKeyFromCompactionLogTable(keysToRemove);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    private synchronized Pair<Set<String>, List<byte[]>> getOlderFileNodes() {
        long compactionLogPruneStartTime = System.currentTimeMillis();
        HashSet compactionNodes = new HashSet();
        ArrayList<byte[]> keysToRemove = new ArrayList<byte[]>();
        try (ManagedRocksIterator managedRocksIterator = new ManagedRocksIterator(((RocksDB)this.activeRocksDB.get()).newIterator(this.compactionLogTableCFHandle));){
            ((RocksIterator)managedRocksIterator.get()).seekToFirst();
            while (((RocksIterator)managedRocksIterator.get()).isValid()) {
                CompactionLogEntry compactionLogEntry = CompactionLogEntry.getFromProtobuf(HddsProtos.CompactionLogEntryProto.parseFrom((byte[])((RocksIterator)managedRocksIterator.get()).value()));
                if (compactionLogPruneStartTime - compactionLogEntry.getCompactionTime() < this.maxAllowedTimeInDag) {
                    break;
                }
                compactionLogEntry.getInputFileInfoList().forEach(inputFileInfo -> compactionNodes.add(inputFileInfo.getFileName()));
                keysToRemove.add(((RocksIterator)managedRocksIterator.get()).key());
                ((RocksIterator)managedRocksIterator.get()).next();
            }
        }
        catch (InvalidProtocolBufferException exception) {
            throw new RuntimeException(exception);
        }
        return Pair.of(compactionNodes, keysToRemove);
    }

    private synchronized void removeKeyFromCompactionLogTable(List<byte[]> keysToRemove) {
        try {
            for (byte[] key : keysToRemove) {
                ((RocksDB)this.activeRocksDB.get()).delete(this.compactionLogTableCFHandle, key);
            }
        }
        catch (RocksDBException exception) {
            throw new RuntimeException(exception);
        }
    }

    private void removeSstFiles(Set<String> sstFileNodes) {
        for (String sstFileNode : sstFileNodes) {
            File file = new File(this.sstBackupDir + "/" + sstFileNode + SST_FILE_EXTENSION);
            try {
                Files.deleteIfExists(file.toPath());
            }
            catch (IOException exception) {
                LOG.warn("Failed to delete SST file: " + sstFileNode, (Throwable)exception);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<String> pruneSstFileNodesFromDag(Set<String> sstFileNodes) {
        HashSet<CompactionNode> startNodes = new HashSet<CompactionNode>();
        for (String sstFileNode : sstFileNodes) {
            CompactionNode infileNode = this.compactionDag.getCompactionNode(sstFileNode);
            if (infileNode == null) {
                LOG.warn("Compaction node doesn't exist for sstFile: {}.", (Object)sstFileNode);
                continue;
            }
            startNodes.add(infileNode);
        }
        RocksDBCheckpointDiffer rocksDBCheckpointDiffer = this;
        synchronized (rocksDBCheckpointDiffer) {
            return this.compactionDag.pruneNodesFromDag(startNodes);
        }
    }

    @VisibleForTesting
    Set<String> pruneBackwardDag(MutableGraph<CompactionNode> backwardDag, Set<CompactionNode> startNodes) {
        return this.compactionDag.pruneBackwardDag(backwardDag, startNodes);
    }

    @VisibleForTesting
    Set<String> pruneForwardDag(MutableGraph<CompactionNode> forwardDag, Set<CompactionNode> startNodes) {
        return this.compactionDag.pruneForwardDag(forwardDag, startNodes);
    }

    private long getSnapshotCreationTimeFromLogLine(String logLine) {
        String line = logLine.substring(COMPACTION_LOG_SEQ_NUM_LINE_PREFIX.length());
        String[] splits = line.split(SPACE_DELIMITER);
        Preconditions.checkArgument((splits.length == 3 ? 1 : 0) != 0, (Object)"Snapshot info log statement has more than expected parameters.");
        return Long.parseLong(splits[2]);
    }

    public String getSSTBackupDir() {
        return this.sstBackupDir;
    }

    public String getCompactionLogDir() {
        return this.compactionLogDir;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void pruneSstFiles() {
        Set<String> nonLeafSstFiles;
        if (!this.shouldRun()) {
            return;
        }
        RocksDBCheckpointDiffer rocksDBCheckpointDiffer = this;
        synchronized (rocksDBCheckpointDiffer) {
            nonLeafSstFiles = this.compactionDag.getForwardCompactionDAG().nodes().stream().filter(node -> !this.compactionDag.getForwardCompactionDAG().successors(node).isEmpty()).map(node -> node.getFileName()).collect(Collectors.toSet());
        }
        if (CollectionUtils.isNotEmpty(nonLeafSstFiles)) {
            LOG.info("Removing SST files: {} as part of SST file pruning.", nonLeafSstFiles);
        }
        try (BootstrapStateHandler.Lock lock = this.getBootstrapStateLock().lock();){
            this.removeSstFiles(nonLeafSstFiles);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void pruneSstFileValues() {
        if (!this.shouldRun()) {
            return;
        }
        long batchStartTime = System.nanoTime();
        int filesPrunedInBatch = 0;
        int filesSkippedInBatch = 0;
        int batchCounter = 0;
        Path sstBackupDirPath = Paths.get(this.sstBackupDir, new String[0]);
        Path prunedSSTFilePath = sstBackupDirPath.resolve(PRUNED_SST_FILE_TEMP);
        try (ManagedOptions managedOptions = new ManagedOptions();
             ManagedEnvOptions envOptions = new ManagedEnvOptions();){
            byte[] compactionLogEntryKey;
            while ((compactionLogEntryKey = this.pruneQueue.peek()) != null && ++batchCounter <= this.pruneSSTFileBatchSize) {
                RocksDBCheckpointDiffer rocksDBCheckpointDiffer = this;
                synchronized (rocksDBCheckpointDiffer) {
                    CompactionLogEntry compactionLogEntry;
                    try {
                        compactionLogEntry = (CompactionLogEntry)CompactionLogEntry.getCodec().fromPersistedFormat(((RocksDB)this.activeRocksDB.get()).get(this.compactionLogTableCFHandle, compactionLogEntryKey));
                    }
                    catch (RocksDBException ex) {
                        throw new RocksDatabaseException("Failed to get compaction log entry.", (Exception)((Object)ex));
                    }
                    boolean shouldUpdateTable = false;
                    List<CompactionFileInfo> fileInfoList = compactionLogEntry.getInputFileInfoList();
                    ArrayList<CompactionFileInfo> updatedFileInfoList = new ArrayList<CompactionFileInfo>();
                    for (CompactionFileInfo fileInfo : fileInfoList) {
                        if (fileInfo.isPruned()) {
                            updatedFileInfoList.add(fileInfo);
                            continue;
                        }
                        Path sstFilePath = sstBackupDirPath.resolve(fileInfo.getFileName() + SST_FILE_EXTENSION);
                        if (Files.notExists(sstFilePath, new LinkOption[0])) {
                            LOG.debug("Skipping pruning SST file {} as it does not exist in backup directory.", (Object)sstFilePath);
                            updatedFileInfoList.add(fileInfo);
                            ++filesSkippedInBatch;
                            continue;
                        }
                        Files.deleteIfExists(prunedSSTFilePath);
                        this.removeValueFromSSTFile(managedOptions, envOptions, sstFilePath.toFile().getAbsolutePath(), prunedSSTFilePath.toFile().getAbsolutePath());
                        try (BootstrapStateHandler.Lock lock = this.getBootstrapStateLock().lock();){
                            Files.move(prunedSSTFilePath, sstFilePath, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
                        }
                        shouldUpdateTable = true;
                        fileInfo.setPruned();
                        updatedFileInfoList.add(fileInfo);
                        LOG.debug("Completed pruning OMKeyInfo from {}", (Object)sstFilePath);
                        ++filesPrunedInBatch;
                    }
                    if (shouldUpdateTable) {
                        CompactionLogEntry.Builder builder = compactionLogEntry.toBuilder();
                        builder.updateInputFileInfoList(updatedFileInfoList);
                        try {
                            ((RocksDB)this.activeRocksDB.get()).put(this.compactionLogTableCFHandle, compactionLogEntryKey, builder.build().getProtobuf().toByteArray());
                        }
                        catch (RocksDBException ex) {
                            throw new RocksDatabaseException("Failed to update the compaction log table for entry: " + compactionLogEntry, (Exception)((Object)ex));
                        }
                    }
                }
                this.pruneQueue.poll();
            }
        }
        catch (IOException | InterruptedException e) {
            LOG.error("Could not prune source OMKeyInfo from backup SST files.", (Throwable)e);
            this.sstFilePruningMetrics.incrPruningFailures();
        }
        finally {
            LOG.info("Completed pruning OMKeyInfo from backup SST files in {}ms.", (Object)((System.nanoTime() - batchStartTime) / 1000000L));
            this.sstFilePruningMetrics.updateBatchLevelMetrics(filesPrunedInBatch, filesSkippedInBatch, batchCounter, this.pruneQueue.size());
        }
    }

    private void removeValueFromSSTFile(ManagedOptions options, ManagedEnvOptions envOptions, String sstFilePath, String prunedFilePath) throws IOException {
        try (ManagedRawSSTFileReader sstFileReader = new ManagedRawSSTFileReader(options, sstFilePath, 0x200000);
             ManagedRawSSTFileIterator itr = sstFileReader.newIterator(keyValue -> Pair.of((Object)keyValue.getKey(), (Object)keyValue.getType()), null, null);
             ManagedSstFileWriter sstFileWriter = new ManagedSstFileWriter((EnvOptions)envOptions, (Options)options);){
            sstFileWriter.open(prunedFilePath);
            while (itr.hasNext()) {
                Pair keyValue2 = (Pair)itr.next();
                if ((Integer)keyValue2.getValue() == 0) {
                    sstFileWriter.delete((byte[])keyValue2.getKey());
                    continue;
                }
                sstFileWriter.put((byte[])keyValue2.getKey(), ArrayUtils.EMPTY_BYTE_ARRAY);
            }
            sstFileWriter.finish();
        }
        catch (RocksDBException ex) {
            throw new RocksDatabaseException("Failed to write pruned entries for " + sstFilePath, (Exception)((Object)ex));
        }
    }

    public boolean shouldRun() {
        return !this.suspended.get();
    }

    @VisibleForTesting
    public ConcurrentMap<String, CompactionNode> getCompactionNodeMap() {
        return this.compactionDag.getCompactionMap();
    }

    @VisibleForTesting
    public void resume() {
        this.suspended.set(false);
    }

    @VisibleForTesting
    public void suspend() {
        this.suspended.set(true);
    }

    public BootstrapStateHandler.Lock getBootstrapStateLock() {
        return this.lock;
    }

    private Map<String, CompactionFileInfo> toFileInfoList(List<String> sstFiles, RocksDB db) {
        if (CollectionUtils.isEmpty(sstFiles)) {
            return Collections.emptyMap();
        }
        Map liveFileMetaDataMap = ManagedRocksDB.getLiveMetadataForSSTFiles((RocksDB)db);
        HashMap<String, CompactionFileInfo> response = new HashMap<String, CompactionFileInfo>();
        for (String sstFile : sstFiles) {
            String fileName = FilenameUtils.getBaseName((String)sstFile);
            CompactionFileInfo fileInfo = new CompactionFileInfo.Builder(fileName).setValues((LiveFileMetaData)liveFileMetaDataMap.get(fileName)).build();
            response.put(sstFile, fileInfo);
        }
        return response;
    }

    ConcurrentMap<String, CompactionFileInfo> getInflightCompactions() {
        return this.inflightCompactions;
    }

    @VisibleForTesting
    public SSTFilePruningMetrics getPruningMetrics() {
        return this.sstFilePruningMetrics;
    }

    static {
        RocksDB.loadLibrary();
    }

    static class NodeComparator
    implements Comparator<CompactionNode>,
    Serializable {
        NodeComparator() {
        }

        @Override
        public int compare(CompactionNode a, CompactionNode b) {
            return a.getFileName().compareToIgnoreCase(b.getFileName());
        }

        @Override
        public Comparator<CompactionNode> reversed() {
            return null;
        }
    }

    public static class RocksDBCheckpointDifferHolder {
        private static final ConcurrentMap<String, RocksDBCheckpointDiffer> INSTANCE_MAP = new ConcurrentHashMap<String, RocksDBCheckpointDiffer>();

        public static RocksDBCheckpointDiffer getInstance(String metadataDirName, String sstBackupDirName, String compactionLogDirName, String activeDBLocationName, ConfigurationSource configuration) {
            return INSTANCE_MAP.computeIfAbsent(metadataDirName, key -> new RocksDBCheckpointDiffer(metadataDirName, sstBackupDirName, compactionLogDirName, activeDBLocationName, configuration));
        }

        public static void invalidateCacheEntry(String cacheKey) {
            IOUtils.close((Logger)LOG, (AutoCloseable[])new AutoCloseable[]{(AutoCloseable)INSTANCE_MAP.remove(cacheKey)});
        }
    }
}

