/*
 * Decompiled with CFR 0.152.
 */
package org.apache.streampark.console.base.util;

import com.fasterxml.jackson.annotation.JsonIgnore;
import java.io.File;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.Normalizer;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.apache.streampark.common.conf.CommonConfig;
import org.apache.streampark.common.conf.InternalConfigHolder;
import org.apache.streampark.common.conf.InternalOption;
import org.apache.streampark.common.util.Utils;
import org.apache.streampark.console.base.util.WebUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MavenUtils {
    private static final Logger log = LoggerFactory.getLogger(MavenUtils.class);
    private static final int MAX_BUILD_ARGS_LENGTH = 2048;
    private static final int MAX_MAVEN_ARG_LENGTH = 512;
    private static final Pattern COMMAND_INJECTION_PATTERN = Pattern.compile("(`[^`]*`)|(\\$\\([^)]*\\))|(\\$\\{[^}]*})|([;&|><])|(\\\\[nrt])|([\\r\\n\\t])|(\\\\x[0-9a-fA-F]{2})|(%[0-9a-fA-F]{2})|(\\\\u[0-9a-fA-F]{4})", 10);
    private static final Set<String> ALLOWED_MAVEN_ARGS = Collections.unmodifiableSet(new HashSet<String>(Arrays.asList("-D", "--define", "-P", "--activate-profiles", "-q", "--quiet", "-X", "--debug", "-e", "--errors", "-f", "--file", "-s", "--settings", "-t", "--toolchains", "-T", "--threads", "-B", "--batch-mode", "-V", "--show-version", "-U", "--update-snapshots", "-N", "--non-recursive", "-C", "--strict-checksums", "-c", "--lax-checksums", "-o", "--offline", "--no-snapshot-updates", "--fail-at-end", "--fail-fast", "--fail-never", "--resume-from", "--projects", "--also-make", "--also-make-dependents", "clean", "compile", "test", "package", "install", "deploy", "validate", "initialize", "generate-sources", "process-sources", "generate-resources", "process-resources", "process-classes", "generate-test-sources", "process-test-sources", "generate-test-resources", "process-test-resources", "test-compile", "process-test-classes", "prepare-package", "pre-integration-test", "integration-test", "post-integration-test", "verify", "install", "deploy")));
    private static final Set<String> ALLOWED_SYSTEM_PROPERTIES = Collections.unmodifiableSet(new HashSet<String>(Arrays.asList("skipTests", "maven.test.skip", "maven.javadoc.skip", "maven.source.skip", "project.build.sourceEncoding", "project.reporting.outputEncoding", "maven.compiler.source", "maven.compiler.target", "maven.compiler.release", "flink.version", "scala.version", "hadoop.version", "kafka.version", "java.version", "encoding", "file.encoding")));

    @JsonIgnore
    public static String getMavenArgs(String buildArgs) {
        try {
            String setting;
            String validatedBuildArgs;
            StringBuilder mvnArgBuffer = new StringBuilder(" clean package -DskipTests ");
            if (StringUtils.isNotBlank((CharSequence)buildArgs) && StringUtils.isNotBlank((CharSequence)(validatedBuildArgs = MavenUtils.validateAndSanitizeBuildArgs(buildArgs.trim())))) {
                mvnArgBuffer.append(validatedBuildArgs);
            }
            if (StringUtils.isNotBlank((CharSequence)(setting = (String)InternalConfigHolder.get((InternalOption)CommonConfig.MAVEN_SETTINGS_PATH())))) {
                String validatedSettingsPath = MavenUtils.validateAndSanitizeSettingsPath(setting);
                mvnArgBuffer.append(" --settings ").append(validatedSettingsPath);
            }
            String mvnArgs = mvnArgBuffer.toString();
            MavenUtils.validateFinalMavenCommand(mvnArgs);
            String mvn = MavenUtils.getSecureMavenExecutable();
            log.info("\ud83d\udd27 Secure Maven command prepared: {} {}", (Object)MavenUtils.sanitizeForLogging(mvn), (Object)MavenUtils.sanitizeForLogging(mvnArgs));
            return mvn.concat(mvnArgs);
        }
        catch (SecurityException e) {
            log.error("\ud83d\udea8 Security validation failed for Maven build: {}", (Object)e.getMessage());
            throw new IllegalArgumentException("Maven build security validation failed: " + e.getMessage(), e);
        }
        catch (Exception e) {
            log.error("\u274c Failed to prepare Maven command: {}", (Object)e.getMessage());
            throw new IllegalArgumentException("Failed to prepare Maven command: " + e.getMessage(), e);
        }
    }

    private static String validateAndSanitizeSettingsPath(String settingsPath) {
        if (StringUtils.isBlank((CharSequence)settingsPath)) {
            MavenUtils.logSecurityEvent("ERROR", "PATH_VALIDATION", "Settings path is blank", "");
            throw new SecurityException("Settings path cannot be blank");
        }
        MavenUtils.logSecurityEvent("INFO", "PATH_VALIDATION", "Starting Maven settings path validation", settingsPath);
        try {
            Path normalizedPath = Paths.get(settingsPath, new String[0]).normalize();
            String pathString = normalizedPath.toString();
            if (pathString.contains("..") || pathString.contains("~")) {
                MavenUtils.logSecurityEvent("VIOLATION", "PATH_TRAVERSAL_DETECTION", "Path traversal attempt detected in settings path", settingsPath);
                throw new SecurityException("Path traversal detected in settings path");
            }
            File file = normalizedPath.toFile();
            if (!file.exists()) {
                throw new SecurityException(String.format("Maven settings file does not exist: %s", MavenUtils.sanitizeForLogging(pathString)));
            }
            if (!file.isFile()) {
                throw new SecurityException(String.format("Maven settings path is not a file: %s", MavenUtils.sanitizeForLogging(pathString)));
            }
            if (!file.canRead()) {
                throw new SecurityException(String.format("Maven settings file is not readable: %s", MavenUtils.sanitizeForLogging(pathString)));
            }
            if (file.length() > 0xA00000L) {
                throw new SecurityException("Maven settings file is too large (>10MB)");
            }
            MavenUtils.logSecurityEvent("SUCCESS", "PATH_VALIDATION", "Maven settings path validation completed", pathString);
            return pathString;
        }
        catch (InvalidPathException e) {
            MavenUtils.logSecurityEvent("VIOLATION", "PATH_SYNTAX_ERROR", "Invalid path syntax in settings path", settingsPath);
            throw new SecurityException("Invalid path syntax in settings path", e);
        }
    }

    private static void validateFinalMavenCommand(String mvnArgs) {
        MavenUtils.logSecurityEvent("INFO", "FINAL_COMMAND_VALIDATION", "Starting final Maven command validation", mvnArgs);
        if (mvnArgs.contains("\n") || mvnArgs.contains("\r")) {
            MavenUtils.logSecurityEvent("VIOLATION", "CONTROL_CHARACTER_DETECTION", "Control characters detected in maven command", mvnArgs);
            throw new SecurityException("Control characters detected in maven build parameters");
        }
        Matcher dangerousMatcher = COMMAND_INJECTION_PATTERN.matcher(mvnArgs);
        if (dangerousMatcher.find()) {
            String dangerousPattern = dangerousMatcher.group();
            MavenUtils.logSecurityEvent("VIOLATION", "INJECTION_DETECTION", "Command injection pattern in final Maven command", dangerousPattern);
            throw new SecurityException("Command injection pattern detected in final Maven command");
        }
        if (mvnArgs.length() > 2148) {
            MavenUtils.logSecurityEvent("VIOLATION", "COMMAND_LENGTH_EXCEEDED", String.format("Maven command exceeds maximum length: %d", mvnArgs.length()), mvnArgs);
            throw new SecurityException("Maven command exceeds maximum allowed length");
        }
        MavenUtils.logSecurityEvent("SUCCESS", "FINAL_COMMAND_VALIDATION", "Final Maven command validation completed", mvnArgs);
    }

    private static String getSecureMavenExecutable() {
        boolean windows = Utils.isWindows();
        String mvn = windows ? "mvn.cmd" : "mvn";
        String mavenHome = MavenUtils.validateMavenHomeEnvironment();
        boolean useWrapper = true;
        if (mavenHome != null) {
            mvn = mavenHome + "/bin/" + mvn;
            try {
                ProcessBuilder pb = new ProcessBuilder(mvn, "--version");
                pb.environment().clear();
                Process process = pb.start();
                boolean finished = process.waitFor(10L, TimeUnit.SECONDS);
                if (finished && process.exitValue() == 0) {
                    useWrapper = false;
                    log.info("\u2705 Validated system Maven installation: {}", (Object)MavenUtils.sanitizeForLogging(mvn));
                } else {
                    log.warn("\u26a0\ufe0f  System Maven validation failed, using wrapper");
                }
            }
            catch (Exception e) {
                log.warn("\u26a0\ufe0f  Maven validation error: {}, using wrapper", (Object)e.getMessage());
            }
        }
        if (useWrapper) {
            String wrapperPath = WebUtils.getAppHome().concat(windows ? "/bin/mvnw.cmd" : "/bin/mvnw");
            File wrapperFile = new File(wrapperPath);
            if (!wrapperFile.exists() || !wrapperFile.canExecute()) {
                throw new SecurityException("Maven wrapper not found or not executable: " + wrapperPath);
            }
            mvn = wrapperPath;
            log.info("\u2705 Using secure Maven wrapper: {}", (Object)MavenUtils.sanitizeForLogging(mvn));
        }
        return mvn;
    }

    private static String validateMavenHomeEnvironment() {
        String mavenHome = System.getenv("M2_HOME");
        if (mavenHome == null) {
            mavenHome = System.getenv("MAVEN_HOME");
        }
        if (mavenHome == null) {
            return null;
        }
        try {
            Path normalizedPath = Paths.get(mavenHome, new String[0]).normalize();
            String pathString = normalizedPath.toString();
            if (pathString.contains("..") || pathString.contains("~")) {
                log.warn("\u26a0\ufe0f  Suspicious Maven home path, ignoring: {}", (Object)MavenUtils.sanitizeForLogging(mavenHome));
                return null;
            }
            File mavenDir = normalizedPath.toFile();
            if (!mavenDir.exists() || !mavenDir.isDirectory()) {
                log.warn("\u26a0\ufe0f  Invalid Maven home directory, ignoring: {}", (Object)MavenUtils.sanitizeForLogging(pathString));
                return null;
            }
            return pathString;
        }
        catch (InvalidPathException e) {
            log.warn("\u26a0\ufe0f  Invalid Maven home path syntax, ignoring: {}", (Object)MavenUtils.sanitizeForLogging(mavenHome));
            return null;
        }
    }

    private static void logSecurityEvent(String eventType, String operation, String details, String sensitiveData) {
        String sanitizedData = MavenUtils.sanitizeForLogging(sensitiveData);
        switch (eventType.toUpperCase()) {
            case "SUCCESS": {
                log.info("\u2705 [SECURITY-AUDIT] {} - {}: {} | Data: {}", new Object[]{eventType, operation, details, sanitizedData});
                break;
            }
            case "WARNING": {
                log.warn("\u26a0\ufe0f  [SECURITY-AUDIT] {} - {}: {} | Data: {}", new Object[]{eventType, operation, details, sanitizedData});
                break;
            }
            case "VIOLATION": 
            case "ERROR": {
                log.error("\ud83d\udea8 [SECURITY-AUDIT] {} - {}: {} | Data: {}", new Object[]{eventType, operation, details, sanitizedData});
                break;
            }
            default: {
                log.info("\u2139\ufe0f  [SECURITY-AUDIT] {} - {}: {} | Data: {}", new Object[]{eventType, operation, details, sanitizedData});
            }
        }
        log.info("SECURITY_EVENT_JSON: {{\"timestamp\":\"{}\",\"event_type\":\"{}\",\"operation\":\"{}\",\"details\":\"{}\",\"source\":\"Project.getMavenArgs\"}}", new Object[]{Instant.now().toString(), eventType, operation, details.replace("\"", "\\\"")});
    }

    private static String validateAndSanitizeBuildArgs(String buildArgs) {
        Matcher dangerousMatcher;
        if (StringUtils.isBlank((CharSequence)buildArgs)) {
            MavenUtils.logSecurityEvent("SUCCESS", "BUILD_VALIDATION", "Empty build arguments processed", "");
            return "";
        }
        MavenUtils.logSecurityEvent("INFO", "BUILD_VALIDATION", "Starting validation of build arguments", buildArgs);
        if (buildArgs.length() > 2048) {
            MavenUtils.logSecurityEvent("VIOLATION", "LENGTH_VALIDATION", String.format("Build arguments exceed maximum length. Length: %d, Limit: %d", buildArgs.length(), 2048), buildArgs);
            throw new SecurityException("Build arguments exceed maximum allowed length: 2048");
        }
        String normalizedArgs = MavenUtils.normalizeAndDecode(buildArgs);
        if (!normalizedArgs.equals(buildArgs)) {
            MavenUtils.logSecurityEvent("WARNING", "ENCODING_DETECTION", "Encoding or Unicode normalization applied to build arguments", String.format("Original: %s | Normalized: %s", buildArgs, normalizedArgs));
        }
        if ((dangerousMatcher = COMMAND_INJECTION_PATTERN.matcher(normalizedArgs)).find()) {
            String dangerousPattern = dangerousMatcher.group();
            MavenUtils.logSecurityEvent("VIOLATION", "INJECTION_DETECTION", "Command injection pattern detected in build arguments", String.format("Pattern: %s | Full args: %s", dangerousPattern, normalizedArgs));
            throw new SecurityException("Dangerous command injection pattern detected: " + MavenUtils.sanitizeForLogging(dangerousPattern));
        }
        String validatedArgs = MavenUtils.validateArgumentsAgainstWhitelist(normalizedArgs);
        MavenUtils.logSecurityEvent("SUCCESS", "BUILD_VALIDATION", "Build arguments validation completed successfully", validatedArgs);
        return validatedArgs;
    }

    private static String normalizeAndDecode(String input) {
        String normalized = Normalizer.normalize(input, Normalizer.Form.NFC);
        normalized = normalized.replace("%20", " ").replace("%3B", ";").replace("%7C", "|").replace("%26", "&").replace("%3E", ">").replace("%3C", "<").replace("%24", "$").replace("%60", "`");
        return normalized;
    }

    private static String validateArgumentsAgainstWhitelist(String args) {
        String[] argArray = args.trim().split("\\s+");
        StringBuilder validatedArgs = new StringBuilder();
        for (int i = 0; i < argArray.length; ++i) {
            String arg = argArray[i].trim();
            if (StringUtils.isBlank((CharSequence)arg)) continue;
            if (arg.length() > 512) {
                log.error("\ud83d\udea8 Security Alert: Individual argument exceeds length limit: {}", (Object)MavenUtils.sanitizeForLogging(arg));
                throw new SecurityException("Individual argument exceeds maximum length: 512");
            }
            if (arg.startsWith("-D")) {
                MavenUtils.validateSystemProperty(arg, i < argArray.length - 1 ? argArray[i + 1] : null);
                validatedArgs.append(arg).append(" ");
                if (i >= argArray.length - 1 || argArray[i + 1].startsWith("-")) continue;
                validatedArgs.append(argArray[++i]).append(" ");
                continue;
            }
            if (arg.startsWith("--define")) {
                MavenUtils.validateSystemProperty(arg, null);
                validatedArgs.append(arg).append(" ");
                continue;
            }
            if (ALLOWED_MAVEN_ARGS.contains(arg)) {
                validatedArgs.append(arg).append(" ");
                continue;
            }
            if (MavenUtils.isValidMavenPhaseOrGoal(arg)) {
                validatedArgs.append(arg).append(" ");
                continue;
            }
            MavenUtils.logSecurityEvent("VIOLATION", "WHITELIST_VALIDATION", "Unauthorized Maven argument detected", arg);
            throw new SecurityException("Unauthorized Maven argument: " + MavenUtils.sanitizeForLogging(arg));
        }
        return validatedArgs.toString().trim();
    }

    private static void validateSystemProperty(String arg, String nextArg) {
        String propertyDefinition = arg;
        if (arg.equals("-D") && nextArg != null) {
            propertyDefinition = nextArg;
        } else if (arg.startsWith("-D")) {
            propertyDefinition = arg.substring(2);
        } else if (arg.startsWith("--define=")) {
            propertyDefinition = arg.substring(9);
        }
        String propertyName = propertyDefinition.split("=")[0];
        if (!ALLOWED_SYSTEM_PROPERTIES.contains(propertyName) && !MavenUtils.isValidSystemPropertyPattern(propertyName)) {
            MavenUtils.logSecurityEvent("VIOLATION", "SYSTEM_PROPERTY_VALIDATION", "Unauthorized system property detected", propertyName);
            throw new SecurityException("Unauthorized system property: " + MavenUtils.sanitizeForLogging(propertyName));
        }
    }

    private static boolean isValidSystemPropertyPattern(String propertyName) {
        List<String> safePatterns = Arrays.asList("maven.", "project.", "flink.", "scala.", "hadoop.", "kafka.", "java.", "user.", "file.", "encoding");
        return safePatterns.stream().anyMatch(propertyName::startsWith) && propertyName.matches("^[a-zA-Z0-9._-]+$");
    }

    private static boolean isValidMavenPhaseOrGoal(String arg) {
        return arg.matches("^[a-zA-Z0-9:._-]+$") && arg.length() <= 50;
    }

    private static String sanitizeForLogging(String input) {
        if (input == null) {
            return "null";
        }
        String sanitized = input.replaceAll("(password|pwd|secret|token|key)=\\S*", "$1=***").replaceAll("(`[^`]*`)", "***BACKTICK_COMMAND***").replaceAll("(\\$\\([^)]*\\))", "***COMMAND_SUBSTITUTION***").replaceAll("(\\$\\{[^}]*})", "***VARIABLE_SUBSTITUTION***");
        if (sanitized.length() > 100) {
            sanitized = sanitized.substring(0, 100) + "...";
        }
        return sanitized;
    }
}

