/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hyracks.api.util;

import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.Serializable;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Objects;
import java.util.stream.Collectors;
import org.apache.commons.io.FileUtils;
import org.apache.hyracks.api.exceptions.ErrorCode;
import org.apache.hyracks.api.exceptions.HyracksDataException;
import org.apache.hyracks.api.io.FileReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class IoUtil {
    public static final String FILE_NOT_FOUND_MSG = "Deleting non-existing file!";
    public static final FilenameFilter NO_OP_FILTER = (dir, name) -> true;
    private static final Logger LOGGER = LogManager.getLogger();

    private IoUtil() {
    }

    public static void delete(Path filePath) throws HyracksDataException {
        IoUtil.delete(filePath.toFile());
    }

    public static void delete(FileReference fileRef) throws HyracksDataException {
        IoUtil.delete(fileRef.getFile());
    }

    public static void delete(File file) throws HyracksDataException {
        try {
            if (file.isDirectory()) {
                if (!file.exists()) {
                    return;
                }
                if (!FileUtils.isSymlink((File)file)) {
                    IoUtil.cleanDirectory(file);
                }
            }
            Files.delete(file.toPath());
        }
        catch (FileNotFoundException | NoSuchFileException e) {
            LOGGER.warn(() -> "Deleting non-existing file!: " + e.getMessage(), (Throwable)e);
        }
        catch (IOException e) {
            throw HyracksDataException.create(ErrorCode.CANNOT_DELETE_FILE, (Throwable)e, new Serializable[]{file.getAbsolutePath()});
        }
    }

    public static void create(FileReference fileRef) throws HyracksDataException {
        if (fileRef.getFile().exists()) {
            throw HyracksDataException.create(ErrorCode.FILE_ALREADY_EXISTS, new Serializable[]{fileRef.getAbsolutePath()});
        }
        fileRef.getFile().getParentFile().mkdirs();
        try {
            if (!fileRef.getFile().createNewFile()) {
                throw HyracksDataException.create(ErrorCode.FILE_ALREADY_EXISTS, new Serializable[]{fileRef.getAbsolutePath()});
            }
        }
        catch (IOException e) {
            throw HyracksDataException.create(ErrorCode.CANNOT_CREATE_FILE, (Throwable)e, new Serializable[]{fileRef.getAbsolutePath()});
        }
    }

    private static void cleanDirectory(File directory) throws IOException {
        File[] files;
        for (File file : files = IoUtil.verifiedListFiles(directory)) {
            IoUtil.delete(file);
        }
    }

    private static File[] verifiedListFiles(File directory) throws IOException {
        if (!directory.exists()) {
            String message = directory + " does not exist";
            throw new IllegalArgumentException(message);
        }
        if (!directory.isDirectory()) {
            String message = directory + " is not a directory";
            throw new IllegalArgumentException(message);
        }
        File[] files = directory.listFiles();
        if (files == null) {
            throw new IOException("Failed to list contents of " + directory);
        }
        return files;
    }

    public static void flushDirectory(File directory) throws IOException {
        IoUtil.flushDirectory(directory.toPath());
    }

    public static void flushDirectory(Path path) throws IOException {
        if (!Files.isDirectory(path, new LinkOption[0])) {
            throw new IOException("Not a directory: " + path);
        }
        if (Files.getFileStore(path).supportsFileAttributeView("posix")) {
            try (FileChannel ch = FileChannel.open(path, StandardOpenOption.READ);){
                ch.force(true);
            }
        } else if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Unable to flush directory " + path);
        }
    }

    public static Collection<File> getMatchingFiles(Path root, FilenameFilter filter) {
        if (!Files.isDirectory(root, new LinkOption[0])) {
            throw new IllegalArgumentException("Parameter 'root' is not a directory: " + root);
        }
        Objects.requireNonNull(filter);
        ArrayList<File> files = new ArrayList<File>();
        FileFilter dirOrMatchingFileFilter = file -> file.isDirectory() || filter.accept(file, file.getName());
        IoUtil.collectDirFiles(root.toFile(), dirOrMatchingFileFilter, files);
        return files;
    }

    private static void collectDirFiles(File dir, FileFilter filter, Collection<File> files) {
        File[] matchingFiles = dir.listFiles(filter);
        if (matchingFiles != null) {
            for (File file : matchingFiles) {
                if (file.isDirectory()) {
                    IoUtil.collectDirFiles(file, filter, files);
                    continue;
                }
                files.add(file);
            }
        }
    }

    public static String getFileNameFromPath(String path) {
        return path.substring(path.lastIndexOf(47) + 1);
    }

    public static Collection<FileReference> getMatchingChildren(FileReference root, FilenameFilter filter) {
        if (!root.getFile().isDirectory()) {
            throw new IllegalArgumentException("Parameter 'root' is not a directory: " + root);
        }
        Objects.requireNonNull(filter);
        ArrayList<FileReference> files = new ArrayList<FileReference>();
        String[] matchingFiles = root.getFile().list(filter);
        if (matchingFiles != null) {
            files.addAll(Arrays.stream(matchingFiles).map(root::getChild).collect(Collectors.toList()));
        }
        return files;
    }
}

